%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "defs.h"

typedef struct typeentry {
   int cat;            /* Basic types like CHR, UINT, FLT, DBL or
                          Pointer type PTR or
                          Array type ARR */
   int dim;            /* Relevant only for ARR type */
   int ref;            /* ARR: Element type (index in tyoe table)
                          PTR: Element type pointed to (index in type table) */
   int width;          /* sizeof field */
} ttentry;

ttentry TT[4096];      /* At most 4096 types (including the basic types) can be supported */
int nttentry;          /* Number of types defined so far */

typedef struct nameentry {
   char *name;         /* Name of the variable */
   int typeref;        /* Index in the type table */
   int offset;         /* Memory offset in the data segment */
} stentry;

stentry ST[16368];     /* symbol table of all variables */
int nstentry = 0;      /* Count of variables in the symbol table */
int width = 0;         /* Width of all variables in the symbol table */

%}

%union {
   int lexval;
   char *lexstr;
   int typeref;
}

%token VOID UCHR CHR USRT SRT ULNG LNG UINT INT FLT DBL ARR PTR
%token <lexval> NUM
%token <lexstr> ID

%type <typeref> BASIC DIM VT1 VT2 VT3 VT4 VT5 VT6

%%

PROG	: DECLIST
	;

DECLIST	: DECLIST DECL
	| DECL
	;

DECL	: BASIC VT1 LIST ';'
	;

BASIC	: VOID   { $$ = searchbasictype (VOID); }
	| UCHR   { $$ = searchbasictype (UCHR); }
	| CHR    { $$ = searchbasictype (CHR); }
	| USRT   { $$ = searchbasictype (USRT); }
	| SRT    { $$ = searchbasictype (SRT); }
	| ULNG   { $$ = searchbasictype (ULNG); }
	| LNG    { $$ = searchbasictype (LNG); }
	| UINT   { $$ = searchbasictype (UINT); }
	| INT    { $$ = searchbasictype (INT); }
	| FLT    { $$ = searchbasictype (FLT); }
	| DBL    { $$ = searchbasictype (DBL); }
	;

LIST	: LIST ',' VT2 VAR
	| VT3 VAR
	;

VAR	: '*' VT4 VAR
	| ID VT5 DIM   { STadd($1,$3); }
	;

DIM	: '[' NUM ']' VT6 DIM { $$ = addarraytype($2,$5); }
	| { $$ = $<typeref>0; }
	;

VT1	: { $$ = $<typeref>0; } ;
VT2	: { $$ = $<typeref>-2; } ;
VT3	: { $$ = $<typeref>0; } ;
VT4	: { $$ = addpointertype($<typeref>-1); } ;
VT5	: { $$ = $<typeref>-1; } ;
VT6	: { $$ = $<typeref>-3; } ;

%%

void yyerror (char *msg ) {
   fprintf(stderr, "*** %s\n", msg);
   exit(1);
}

int main ( int argc, char *argv[] )
{
   if (argc > 1) yyin = (FILE *)fopen(argv[1],"r");
   addbasictypes();
   yyparse();
   fclose(yyin);
   printalltypes();
   printallsymbols();
   exit(0);
}

void addbasictypes ( )
{
   TT[0].cat  = VOID; TT[0].width  = 0;
   TT[1].cat  = UCHR; TT[1].width  = sizeof(unsigned char);
   TT[2].cat  = CHR;  TT[2].width  = sizeof(char);
   TT[3].cat  = USRT; TT[3].width  = sizeof(unsigned short);
   TT[4].cat  = SRT;  TT[4].width  = sizeof(short);
   TT[5].cat  = ULNG; TT[5].width  = sizeof(unsigned long);
   TT[6].cat  = LNG;  TT[6].width  = sizeof(long);
   TT[7].cat  = UINT; TT[7].width  = sizeof(unsigned int);
   TT[8].cat  = INT;  TT[8].width  = sizeof(int);
   TT[9].cat  = FLT;  TT[9].width  = sizeof(float);
   TT[10].cat = DBL;  TT[10].width = sizeof(double);
   nttentry = 11;
}

int searchbasictype ( int type )
{
   switch (type) {
      case VOID : return 0;
      case UCHR : return 1;
      case CHR  : return 2;
      case USRT : return 3;
      case SRT  : return 4;
      case ULNG : return 5;
      case LNG  : return 6;
      case UINT : return 7;
      case INT  : return 8;
      case FLT  : return 9;
      case DBL  : return 10;
      default:
         fprintf(stderr, "*** Unrecognized basic type: %d\n", type);
         return -1;
   }
}

int addarraytype ( int dim, int ttref )
{
   int i;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == ARR) && (TT[i].ref == ttref) && (TT[i].dim == dim) ) return i;
   }
   i = nttentry;
   ++nttentry;
   TT[i].cat = ARR;
   TT[i].dim = dim;
   TT[i].ref = ttref;
   TT[i].width = dim * TT[ttref].width;
   return i;
}

int addpointertype ( int ttref )
{
   int i, w;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == PTR) && (TT[i].ref == ttref) ) return i;
   }
   i = nttentry;
   ++nttentry;
   TT[i].cat = PTR;
   TT[i].ref = ttref;
   TT[i].width = sizeof(void *);
   return i;
}

void STadd ( char *name, int typeref )
{
   int i, w;

   for (i=0; i<nstentry; ++i) {
      if (!strcmp(ST[i].name,name)) yyerror("Duplicate variable name");
   }
   i = nstentry;
   ++nstentry;
   ST[i].name = strdup(name);
   ST[i].typeref = typeref;
   ST[i].offset = width;
   width += TT[typeref].width;
   width = ((width + 3) / 4) * 4;
}

void printtype ( int i )
{
   if (TT[i].cat == ARR) {
      printf("array(%d,", TT[i].dim); printtype(TT[i].ref); printf(")");
   } else if (TT[i].cat == PTR) {
      printf("pointer("); printtype(TT[i].ref); printf(")");
   } else {
      printbasicname(TT[i].cat);
   }
}

void printalltypes ( )
{
   int i;

   printf("+++ All declarations read\n");
   printf("\n+++ %d types\n", nttentry);
   for (i=0; i<nttentry; ++i) {
      printf("    Type %3d: %8d    ", i, TT[i].width);
      printtype(i);
      printf("\n");
   }
}

void printbasicname ( int i )
{
   switch (i) {
      case VOID:  printf("void"); break;
      case UCHR:  printf("unsigned char"); break;
      case CHR :  printf("char"); break;
      case USRT:  printf("unsigned short"); break;
      case SRT :  printf("short"); break;
      case ULNG:  printf("unsigned long"); break;
      case LNG :  printf("long"); break;
      case UINT:  printf("unsigned int"); break;
      case INT :  printf("int"); break;
      case FLT :  printf("float"); break;
      case DBL :  printf("double"); break;
      default  :  printf("unknown"); break;
   }
}

void printallsymbols ( )
{
   int i;

   printf("\n+++ Symbol table\n");
   for (i=0; i<nstentry; ++i) {
      printf("    %-12s%10d - %-10d  type = %4d = ", ST[i].name, ST[i].offset,
             ST[i].offset + TT[ST[i].typeref].width - 1,
             ST[i].typeref);
      printtype(ST[i].typeref);
      printf("\n");
   }
   printf("    Total width = %d\n", width);
}
