#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "defs.h"
#include "y.tab.h"

ttentry TT[4096];      /* At most 4096 types (including the basic types) can be supported */
int nttentry;          /* Number of types defined so far */

stentry ST[64][1024];  /* ST[0] is the main symbol table. ST[1], ST[2], ... are for structures */
int nstentry[64];      /* Count of variables in each symbol table */
int nst = 1;           /* Number of symbol tables in use */
int width[64];         /* Total width of all variables in the symbol table */
int stref[64];         /* Reference in the type table */

int tmpno = 0;

int instno = -1;
char inst[4096][128], leader[4096];

void yyerror (char *msg ) {
   fprintf(stderr, "\n*** Error: %s\n\n", msg);
   exit(1);
}

int main ( int argc, char *argv[] )
{
   int i;

   if (argc > 1) yyin = (FILE *)fopen(argv[1],"r");
   addbasictypes();
   for (i=0; i<4096; ++i) leader[i] = 0;
   leader[0] = 1;
   yyparse();
   fclose(yyin);
   printallinst();
   exit(0);
}

void addbasictypes ( )
{
   TT[0].cat = INT; TT[0].width = 4;
   TT[1].cat = LNG; TT[1].width = 8;
   TT[2].cat = FLT; TT[2].width = 4;
   TT[3].cat = DBL; TT[3].width = 8;
   nttentry = 4;
}

int searchbasictype ( int type )
{
   switch (type) {
      case INT  : return 0;
      case LNG  : return 1;
      case FLT  : return 2;
      case DBL  : return 3;
      default:
         fprintf(stderr, "*** Unrecognized basic type: %d\n", type);
         return -1;
   }
}

int addarraytype ( int dim, int ttref )
{
   int i;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == ARR) && (TT[i].ref == ttref) && (TT[i].dim == dim) ) return i;
   }
   i = nttentry;
   ++nttentry;
   TT[i].cat = ARR;
   TT[i].dim = dim;
   TT[i].ref = ttref;
   TT[i].width = dim * TT[ttref].width;
   return i;
}


int searchstructtype ( char *name )
{
   int i;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == STRUCT) && (!strcmp(TT[i].name,name)) ) return i;
   }
   yyerror("Undefined structure name");
}

int addstructtype ( char *name )
{
   int i, j, w;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == STRUCT) && (!strcmp(TT[i].name,name)) ) {
         yyerror("Structure name already exists");
      }
   }
   i = nttentry;
   ++nttentry;
   TT[i].cat = STRUCT;
   TT[i].name = strdup(name);
   j = nst;
   ++nst;
   TT[i].ref = j;
   stref[j] = i;
   width[j] = 0;
   return j;
}

void setstructwidth ( int t )
{
   TT[stref[t]].width = width[t];
}

int getstref ( int i )
{
   return stref[i];
}

int getstructref ( int i )
{
   if (TT[i].cat != STRUCT) yyerror("invalid structure reference");
   return TT[i].ref;
}

struct _attr *genattr ( int typeref, int stno )
{
   struct _attr *p;

   p = (struct _attr *)malloc(sizeof(struct _attr));
   p -> typeref = typeref;
   p -> stno = stno;
   return p;
}

void STadd ( char *name, int typeref, int t )
{
   int i, w;

   for (i=0; i<nstentry[t]; ++i) {
      if (!strcmp(ST[t][i].name,name)) yyerror("Duplicate variable name");
   }
   i = nstentry[t];
   ++(nstentry[t]);
   ST[t][i].name = strdup(name);
   ST[t][i].typeref = typeref;
   ST[t][i].offset = width[t];
   width[t] += TT[typeref].width;
   width[t] = ((width[t] + 3) / 4) * 4;
}

int STget ( char *name, int t )
{
   int i;
   char msg[1024];

   for (i=0; i<nstentry[t]; ++i) {
      if (!strcmp(ST[t][i].name,name)) return i;
   }
   sprintf(msg, "*** Undefined name in symbol table %d: %s", t, name);
   yyerror(msg);
}

void printtype ( int i )
{
   if (TT[i].cat == ARR) {
      printf("array(%d,", TT[i].dim); printtype(TT[i].ref); printf(")");
   } else if (TT[i].cat  == STRUCT) {
      printf("struct %s [st = %d]", TT[i].name, TT[i].ref);
   } else {
      printbasicname(TT[i].cat);
   }
}

void printalltypes ( )
{
   int i;

   printf("+++ All declarations read\n");
   printf("\n+++ %d types\n", nttentry);
   for (i=0; i<nttentry; ++i) {
      printf("    Type %3d: %8d    ", i, TT[i].width);
      printtype(i);
      printf("\n");
   }
}

void printbasicname ( int i )
{
   switch (i) {
      case INT : printf("int"); break;
      case LNG : printf("long"); break;
      case FLT : printf("float"); break;
      case DBL : printf("double"); break;
      default  : printf("unknown"); break;
   }
}

void printallsymbols ( )
{
   int i;

   for (i=0; i<nst; ++i) {
      printf("\n+++ Symbol table %d ", i);
      if (i) printf("[struct %s]\n", TT[stref[i]].name);
      else printf("[main]\n");
      printsymbols(i);
   }
}

void printsymbols ( int t )
{
   int i;

   for (i=0; i<nstentry[t]; ++i) {
      printf("    %-12s%10d - %-10d  type = %4d = ", ST[t][i].name, ST[t][i].offset,
             ST[t][i].offset + TT[ST[t][i].typeref].width - 1,
             ST[t][i].typeref);
      printtype(ST[t][i].typeref);
      printf("\n");
   }
   printf("    Total width = %d\n", width[t]);
}

struct _addr *iconst2addr ( int n ) {
   struct _addr *ap;

   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = ICONST;
   ap -> type = 0;
   (ap -> value).iconst = n;
   return ap;
}

struct _addr *fconst2addr ( double x ) {
   struct _addr *ap;

   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = FCONST;
   ap -> type = 3;
   (ap -> value).fconst = x;
   return ap;
}

struct _addr *id2addr ( char *name, int t )
{
   struct _addr *ap;
   int i;

   i = STget(name,t);
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = MEMLOC;
   ap -> type = ST[t][i].typeref;
   (ap -> value).offset = ST[t][i].offset;
   return ap;
}

struct _addr *memload ( struct _addr *AA )
{
   struct _addr *A;

   A = (struct _addr *)malloc(sizeof(struct _addr));
   A -> cat = TEMP;
   A -> type = AA -> type;
   ++tmpno; ++instno;
   (A -> value).temp = tmpno;
   if (AA -> cat == MEMLOC)
      sprintf(inst[instno], "[%s]  t%d = MEM(%d,%d)",
         printtypeshort(A -> type), tmpno, (AA -> value).offset, TT[AA->type].width);
   else if (AA -> cat == MEMLOCTEMP)
      sprintf(inst[instno], "[%s]  t%d = MEM(t%d,%d)",
         printtypeshort(A -> type), tmpno, (AA -> value).toffset, TT[AA->type].width);
   return A;
}

void addr2idx ( struct _addr *A, int w )
{
   if (A -> cat == ICONST) {
      ++tmpno; ++instno;
      sprintf(inst[instno], "[int]  t%d = %d * %d", tmpno, w, (A -> value).iconst);
   } else if (A -> cat == FCONST) {
      ++tmpno; ++instno;
      sprintf(inst[instno], "[int]  t%d = (dbl2int)%.16lf", tmpno, (A -> value).fconst);
      ++tmpno; ++instno;
      sprintf(inst[instno], "[int]  t%d = %d * t%d", tmpno, w, tmpno - 1);
   } else {
      if (A -> type >= 4) yyerror("invalid type in array index");
      if ( (A -> cat == MEMLOC) || (A -> cat == MEMLOCTEMP) ) A = memload(A);
      if (A -> type == 0) {
         ++tmpno; ++instno;
         sprintf(inst[instno], "[int]  t%d = %d * t%d", tmpno, w, (A -> value).temp);
      } else {
         ++tmpno; ++instno;
         sprintf(inst[instno], "[int]  t%d = (%s2int)%d",
            tmpno, printtypeshort(A -> type), (A -> value).temp);
         ++tmpno; ++instno;
         sprintf(inst[instno], "[int]  t%d = %d * t%d", tmpno, w, tmpno - 1);
      }
   }
}

struct _addr *arrid2addr ( char *name, struct _addr *A, int t )
{
   struct _addr *ap;
   int i;

   i = STget(name,t);
   addr2idx(A,TT[TT[ST[t][i].typeref].ref].width);
   ++tmpno; ++instno;
   sprintf(inst[instno], "[int]  t%d = %d + t%d", tmpno, ST[t][i].offset, tmpno - 1);

   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = MEMLOCTEMP;
   ap -> type = TT[ST[t][i].typeref].ref;
   (ap -> value).toffset = tmpno;
   return ap;
}

struct _addr *arr2addr ( struct _addr *A, struct _addr *B )
{
   struct _addr *ap;

   addr2idx(B,TT[TT[A->type].ref].width);
   ++tmpno; ++instno;
   sprintf(inst[instno], "[int]  t%d = t%d + t%d", tmpno, (A -> value).temp, tmpno - 1);
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = MEMLOCTEMP;
   ap -> type = TT[A -> type].ref;
   (ap -> value).toffset = tmpno;
   return ap;
}

char *printoffset ( struct _addr *A )
{
   char *str = malloc(32 * sizeof(char));

   if (A -> cat == MEMLOC) sprintf(str, "%d", (A -> value).offset);
   else if (A -> cat == MEMLOCTEMP) sprintf(str, "t%d", (A -> value).toffset);
   else sprintf(str, "unknown");
   return str;
}

struct _addr *struct2addr ( struct _addr *A, struct _addr *B )
{
   struct _addr *ap;

   ++tmpno; ++instno;
   sprintf(inst[instno], "[int]  t%d = %s + %s", tmpno, printoffset(A), printoffset(B));
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = MEMLOCTEMP;
   ap -> type = B -> type;
   (ap -> value).toffset = tmpno;
   return ap;
}

struct _addr *item2addr ( struct _addr *A )
{
   struct _addr *ap;

   if ((A -> cat == TEMP) || (A -> cat == MEMLOCTEMP)) {
      return A;
   } else if (A -> cat == MEMLOC) {
      ++tmpno; ++instno;
      sprintf(inst[instno], "[%s]  t%d = MEM(%d,%d)",
         printtypeshort(A -> type), tmpno, (A -> value).offset, TT[A->type].width);
      ap = (struct _addr *)malloc(sizeof(struct _addr));
      ap -> cat = TEMP;
      ap -> type = A -> type;
      (ap -> value).temp = tmpno;
      return ap;
   }
}

char *printarg ( struct _addr *A )
{
   char *str = malloc(32 * sizeof(char));

   if (A -> cat == ICONST) sprintf(str, "%d", (A -> value).iconst);
   else if (A -> cat == FCONST) sprintf(str, "%.16lf", (A -> value).fconst);
   else if (A -> cat == TEMP) sprintf(str, "t%d", (A -> value).temp);
   else sprintf(str, "unknown");
   return str;
}

int LCA ( int type1, int type2 )
{
   if (type1 == type2) return type1;
   if ((type1 == 2) && (type2 == 1)) return 3;
   if ((type1 == 1) && (type2 == 2)) return 3;
   return (type1 > type2) ? type1 : type2;
}

char *printtypeshort ( int type )
{
   char *str = malloc(4 * sizeof(char));

   switch (type) {
      case 0: sprintf(str, "int"); break;
      case 1: sprintf(str, "lng"); break;
      case 2: sprintf(str, "flt"); break;
      case 3: sprintf(str, "dbl"); break;
      default : sprintf(str, "???"); break;
   }
   return str;
}

char *typeconv ( int type1 , int type2 )
{
   char *str = malloc(10 * sizeof(char));
   sprintf(str, "(%s2%s)", printtypeshort(type1), printtypeshort(type2));
   return str;
}

struct _addr *widen ( struct _addr *A, int type )
{
   struct _addr *ap;

   if (A -> type == type) return A;
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = TEMP;
   ++tmpno; (ap -> value).temp = tmpno;
   ap -> type = type;
   ++instno;
   sprintf(inst[instno], "[%s]  t%d = %s%s",
      printtypeshort(type), tmpno, typeconv(A->type,type), printarg(A));
   return ap;
}

struct _addr *operation ( struct _addr *AA, char op, struct _addr *BB )
{
   struct _addr *A, *B, *ap;
   int lca;

   A = ( (AA -> cat == MEMLOC) || (AA -> cat == MEMLOCTEMP) ) ? memload(AA) : AA;
   B = ( (BB -> cat == MEMLOC) || (BB -> cat == MEMLOCTEMP) ) ? memload(BB) : BB;
   if (A -> type >= 4) yyerror("invalid first operand in arithmetic operation");
   if (B -> type >= 4) yyerror("invalid second operand in arithmetic operation");
   if ((op == '%') && ((A -> type >= 2) || (B -> type >= 2)) )
      yyerror("% operator on non-integer arguments");
   lca = LCA(A -> type, B -> type);
   A = widen(A,lca);
   B = widen(B,lca);
   ++tmpno; ++instno;
   sprintf(inst[instno], "[%s]  t%d = %s %c %s",
      printtypeshort(lca), tmpno, printarg(A), op, printarg(B));
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = TEMP;
   ap -> type = lca;
   (ap -> value).temp = tmpno;
   return ap;
}

char *printrval ( struct _addr *rval )
{
   char *str = malloc(32 * sizeof(char));

   if (rval -> cat == ICONST) sprintf(str, "%d", (rval -> value).iconst);
   else if (rval -> cat == FCONST) sprintf(str, "%.16lf", (rval -> value).fconst);
   else if (rval -> cat == TEMP) sprintf(str, "t%d", (rval -> value).temp);
   else if (rval -> cat == MEMLOCTEMP) sprintf(str, "t%d", (rval -> value).toffset);
   else sprintf(str, "unknown");
   return str;
}

void memstore ( struct _addr *lval, struct _addr *rval )
{
   if (lval -> type >= 4) yyerror("invalid type of l-value");
   if (rval -> type >= 4) yyerror("invalid type of r-value");
   if (rval -> cat == MEMLOCTEMP) {
      ++tmpno; ++instno;
      sprintf(inst[instno], "[%s]  t%d = MEM(t%d,%d)",
         printtypeshort(rval -> type), tmpno, (rval -> value).toffset, TT[rval->type].width);
      rval -> cat = TEMP;
      (rval -> value).temp = tmpno;
   }
   if (lval -> type != rval -> type) {
      ++tmpno; ++instno;
      sprintf(inst[instno], "[%s]  t%d = %s%s",
         printtypeshort(lval -> type), tmpno, typeconv(rval->type,lval->type), printarg(rval));
      rval -> cat = MEMLOCTEMP;
      rval -> type = lval -> type;
      (rval -> value).toffset = tmpno;
   }
   ++instno;
   if (lval -> cat == MEMLOCTEMP) {
      sprintf(inst[instno], "[%s]  MEM(t%d,%d) = %s",
         printtypeshort(lval->type), (lval -> value).toffset, TT[lval -> type].width, printrval(rval));
   } else if (lval -> cat == MEMLOC) {
      sprintf(inst[instno], "[%s]  MEM(%d,%d) = %s",
         printtypeshort(lval->type), (lval -> value).offset, TT[lval -> type].width, printrval(rval));
   } else yyerror("Bad l-value");
}

void printallinst ( )
{
   int i;

   for (i=0; i<=instno; ++i) {
      if (leader[i]) printf("\n");
      printf("%6d  :  %s\n", i, inst[i]);
   }
   printf("\n%6d:\n", instno+1);
}

struct _node *makelist ( int i )
{
   struct _node *p;

   p = (struct _node *)malloc(sizeof(struct _node));
   p -> inst = i;
   p -> next = NULL;
   return p;
}

struct _node *merge ( struct _node *L1 , struct _node *L2 )
{
   struct _node *p;

   if (L1 == NULL) return L2;
   if (L2 == NULL) return L1;
   p = L1;
   while (p -> next != NULL) p = p -> next;
   p -> next = L2;
   return L1;
}

void backpatch ( struct _node *L , int j )
{
   int i;
   struct _node *p;

   p = L;
   while (p) {
      i = p -> inst;
      sprintf(inst[i] + strlen(inst[i]), "%d", j);
      leader[j] = 1;
      leader[i+1] = 1;
      p = p -> next;
   }
}

int getnextinst ( )
{
   return instno + 1;
}

int getnext2nextinst ( )
{
   return instno + 2;
}

void genexpgoto ( int i )
{
   ++instno;
   sprintf(inst[instno], "       goto %d", i);
   leader[i] = 1;
   leader[instno + 1] = 1;
}

void genuncgoto ( )
{
   ++instno;
   sprintf(inst[instno], "       goto ");
}

char *printrelop ( int op )
{
   switch (op) {
      case EQ: return strdup("==");
      case NE: return strdup("!=");
      case LT: return strdup("<");
      case LE: return strdup("<=");
      case GT: return strdup(">");
      case GE: return strdup(">=");
      default: return strdup("???");
   }
}

struct _tflist *relop2tflist ( struct _addr *AA , int op , struct _addr *BB )
{
   struct _tflist *TF;
   struct _addr *A, *B;
   int lca;

   A = ( (AA -> cat == MEMLOC) || (AA -> cat == MEMLOCTEMP) ) ? memload(AA) : AA;
   B = ( (BB -> cat == MEMLOC) || (BB -> cat == MEMLOCTEMP) ) ? memload(BB) : BB;
   if ( (A -> type >= 4) || (B -> type >= 4) ) yyerror("invalid type in comparison");
   lca = LCA(A -> type, B -> type);
   A = widen(A,lca);
   B = widen(B,lca);
   TF = (struct _tflist *)malloc(sizeof(struct _tflist));
   TF -> tlist = makelist(++instno);
   sprintf(inst[instno], "       if %s %s %s goto ", printrval(A), printrelop(op), printrval(B));
   TF -> flist = makelist(++instno);
   sprintf(inst[instno], "       goto ");
   return TF;
}
