#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "defs.h"
#include "y.tab.h"

ttentry TT[4096];      /* At most 4096 types (including the basic types) can be supported */
int nttentry;          /* Number of types defined so far */

stentry ST[64][1024];  /* ST[0] is the main symbol table. ST[1], ST[2], ... are for structures */
int nstentry[64];      /* Count of variables in each symbol table */
int nst = 1;           /* Number of symbol tables in use */
int width[64];         /* Total width of all variables in the symbol table */
int stref[64];         /* Reference in the type table */

int tmpno = 0;

void yyerror (char *msg ) {
   fprintf(stderr, "\n*** Error: %s\n\n", msg);
   exit(1);
}

int main ( int argc, char *argv[] )
{
   if (argc > 1) yyin = (FILE *)fopen(argv[1],"r");
   addbasictypes();
   yyparse();
   fclose(yyin);
   exit(0);
}

void addbasictypes ( )
{
   TT[0].cat = INT; TT[0].width = 4;
   TT[1].cat = LNG; TT[1].width = 8;
   TT[2].cat = FLT; TT[2].width = 4;
   TT[3].cat = DBL; TT[3].width = 8;
   nttentry = 4;
}

int searchbasictype ( int type )
{
   switch (type) {
      case INT  : return 0;
      case LNG  : return 1;
      case FLT  : return 2;
      case DBL  : return 3;
      default:
         fprintf(stderr, "*** Unrecognized basic type: %d\n", type);
         return -1;
   }
}

int addarraytype ( int dim, int ttref )
{
   int i;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == ARR) && (TT[i].ref == ttref) && (TT[i].dim == dim) ) return i;
   }
   i = nttentry;
   ++nttentry;
   TT[i].cat = ARR;
   TT[i].dim = dim;
   TT[i].ref = ttref;
   TT[i].width = dim * TT[ttref].width;
   return i;
}


int searchstructtype ( char *name )
{
   int i;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == STRUCT) && (!strcmp(TT[i].name,name)) ) return i;
   }
   yyerror("Undefined structure name");
}

int addstructtype ( char *name )
{
   int i, j, w;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == STRUCT) && (!strcmp(TT[i].name,name)) ) {
         yyerror("Structure name already exists");
      }
   }
   i = nttentry;
   ++nttentry;
   TT[i].cat = STRUCT;
   TT[i].name = strdup(name);
   j = nst;
   ++nst;
   TT[i].ref = j;
   stref[j] = i;
   width[j] = 0;
   return j;
}

void setstructwidth ( int t )
{
   TT[stref[t]].width = width[t];
}

int getstref ( int i )
{
   return stref[i];
}

int getstructref ( int i )
{
   if (TT[i].cat != STRUCT) yyerror("invalid structure reference");
   return TT[i].ref;
}

struct _attr *genattr ( int typeref, int stno )
{
   struct _attr *p;

   p = (struct _attr *)malloc(sizeof(struct _attr));
   p -> typeref = typeref;
   p -> stno = stno;
   return p;
}

void STadd ( char *name, int typeref, int t )
{
   int i, w;

   for (i=0; i<nstentry[t]; ++i) {
      if (!strcmp(ST[t][i].name,name)) yyerror("Duplicate variable name");
   }
   i = nstentry[t];
   ++(nstentry[t]);
   ST[t][i].name = strdup(name);
   ST[t][i].typeref = typeref;
   ST[t][i].offset = width[t];
   width[t] += TT[typeref].width;
   width[t] = ((width[t] + 3) / 4) * 4;
}

int STget ( char *name, int t )
{
   int i;
   char msg[1024];

   for (i=0; i<nstentry[t]; ++i) {
      if (!strcmp(ST[t][i].name,name)) return i;
   }
   sprintf(msg, "*** Undefined name in symbol table %d: %s", t, name);
   yyerror(msg);
}

void printtype ( int i )
{
   if (TT[i].cat == ARR) {
      printf("array(%d,", TT[i].dim); printtype(TT[i].ref); printf(")");
   } else if (TT[i].cat  == STRUCT) {
      printf("struct %s [st = %d]", TT[i].name, TT[i].ref);
   } else {
      printbasicname(TT[i].cat);
   }
}

void printalltypes ( )
{
   int i;

   printf("+++ All declarations read\n");
   printf("\n+++ %d types\n", nttentry);
   for (i=0; i<nttentry; ++i) {
      printf("    Type %3d: %8d    ", i, TT[i].width);
      printtype(i);
      printf("\n");
   }
}

void printbasicname ( int i )
{
   switch (i) {
      case INT : printf("int"); break;
      case LNG : printf("long"); break;
      case FLT : printf("float"); break;
      case DBL : printf("double"); break;
      default  : printf("unknown"); break;
   }
}

void printallsymbols ( )
{
   int i;

   for (i=0; i<nst; ++i) {
      printf("\n+++ Symbol table %d ", i);
      if (i) printf("[struct %s]\n", TT[stref[i]].name);
      else printf("[main]\n");
      printsymbols(i);
   }
}

void printsymbols ( int t )
{
   int i;

   for (i=0; i<nstentry[t]; ++i) {
      printf("    %-12s%10d - %-10d  type = %4d = ", ST[t][i].name, ST[t][i].offset,
             ST[t][i].offset + TT[ST[t][i].typeref].width - 1,
             ST[t][i].typeref);
      printtype(ST[t][i].typeref);
      printf("\n");
   }
   printf("    Total width = %d\n", width[t]);
}

struct _addr *iconst2addr ( int n ) {
   struct _addr *ap;

   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = ICONST;
   ap -> type = 0;
   (ap -> value).iconst = n;
   return ap;
}

struct _addr *fconst2addr ( double x ) {
   struct _addr *ap;

   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = FCONST;
   ap -> type = 3;
   (ap -> value).fconst = x;
   return ap;
}

struct _addr *id2addr ( char *name, int t )
{
   struct _addr *ap;
   int i;

   i = STget(name,t);
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = MEMLOC;
   ap -> type = ST[t][i].typeref;
   (ap -> value).offset = ST[t][i].offset;
   return ap;
}

struct _addr *memload ( struct _addr *AA )
{
   struct _addr *A;

   A = (struct _addr *)malloc(sizeof(struct _addr));
   A -> cat = TEMP;
   A -> type = AA -> type;
   ++tmpno;
   (A -> value).temp = tmpno;
   printf("\t["); printtypeshort(A -> type);
   if (AA -> cat == MEMLOC)
      printf("]\tt%d = MEM(%d,%d)\n", tmpno, (AA -> value).offset, TT[AA->type].width);
   else if (AA -> cat == MEMLOCTEMP)
      printf("]\tt%d = MEM(t%d,%d)\n", tmpno, (AA -> value).toffset, TT[AA->type].width);
   return A;
}

void addr2idx ( struct _addr *A, int w )
{
   if (A -> cat == ICONST) {
      ++tmpno;
      printf("\t[int]\tt%d = %d * %d\n", tmpno, w, (A -> value).iconst);
   } else if (A -> cat == FCONST) {
      ++tmpno;
      printf("\t[int]\tt%d = (dbl2int)%.16lf\n", tmpno, (A -> value).fconst);
      ++tmpno;
      printf("\t[int]\tt%d = %d * t%d\n", tmpno, w, tmpno - 1);
   } else {
      if (A -> type >= 4) yyerror("invalid type in array index");
      if ( (A -> cat == MEMLOC) || (A -> cat == MEMLOCTEMP) ) A = memload(A);
      if (A -> type == 0) {
         ++tmpno;
         printf("\t[int]\tt%d = %d * t%d\n", tmpno, w, (A -> value).temp);
      } else {
         ++tmpno;
         printf("\t[int]\tt%d = (", tmpno); printtypeshort(A -> type);
         printf("2int)t%d\n", (A -> value).temp);
         ++tmpno;
         printf("\t[int]\tt%d = %d * t%d\n", tmpno, w, tmpno - 1);
      }
   }
}

struct _addr *arrid2addr ( char *name, struct _addr *A, int t )
{
   struct _addr *ap;
   int i;

   i = STget(name,t);
   addr2idx(A,TT[TT[ST[t][i].typeref].ref].width);
   ++tmpno;
   printf("\t[int]\tt%d = %d + t%d\n", tmpno, ST[t][i].offset, tmpno - 1);

   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = MEMLOCTEMP;
   ap -> type = TT[ST[t][i].typeref].ref;
   (ap -> value).toffset = tmpno;
   return ap;
}

struct _addr *arr2addr ( struct _addr *A, struct _addr *B )
{
   struct _addr *ap;

   addr2idx(B,TT[TT[A->type].ref].width);
   ++tmpno;
   printf("\t[int]\tt%d = t%d + t%d\n", tmpno, (A -> value).temp, tmpno - 1);
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = MEMLOCTEMP;
   ap -> type = TT[A -> type].ref;
   (ap -> value).toffset = tmpno;
   return ap;
}

struct _addr *struct2addr ( struct _addr *A, struct _addr *B )
{
   struct _addr *ap;

   ++tmpno; printf("\t[int]\tt%d = ", tmpno);
   if (A -> cat == MEMLOC) printf("%d", (A -> value).offset);
   else if (A -> cat == MEMLOCTEMP) printf("t%d", (A -> value).toffset);
   else printf("unknown");
   printf(" + ");
   if (B -> cat == MEMLOC) printf("%d\n", (B -> value).offset);
   else if (B -> cat == MEMLOCTEMP) printf("t%d\n", (B -> value).toffset);
   else printf("unknown\n");
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = MEMLOCTEMP;
   ap -> type = B -> type;
   (ap -> value).toffset = tmpno;
   return ap;
}

struct _addr *item2addr ( struct _addr *A )
{
   struct _addr *ap;

   if ((A -> cat == TEMP) || (A -> cat == MEMLOCTEMP)) {
      return A;
   } else if (A -> cat == MEMLOC) {
      ++tmpno; printf("\t["); printtypeshort(A -> type);
      printf("]\tt%d = MEM(%d,%d)\n", tmpno, (A -> value).offset, TT[A->type].width);
      ap = (struct _addr *)malloc(sizeof(struct _addr));
      ap -> cat = TEMP;
      ap -> type = A -> type;
      (ap -> value).temp = tmpno;
      return ap;
   }
}

void printarg ( struct _addr *A )
{
   if (A -> cat == ICONST) printf("%d", (A -> value).iconst);
   else if (A -> cat == FCONST) printf("%.16lf", (A -> value).fconst);
   else if (A -> cat == TEMP) printf("t%d", (A -> value).temp);
   else printf("unknown");
}

int LCA ( int type1, int type2 )
{
   if (type1 == type2) return type1;
   if ((type1 == 2) && (type2 == 1)) return 3;
   if ((type1 == 1) && (type2 == 2)) return 3;
   return (type1 > type2) ? type1 : type2;
}

void printtypeshort ( int type )
{
   switch (type) {
      case 0: printf("int"); break;
      case 1: printf("lng"); break;
      case 2: printf("flt"); break;
      case 3: printf("dbl"); break;
      default : printf("?%d", type); break;
   }
}

void typeconv ( int type1 , int type2 )
{
   printf("(");
   printtypeshort(type1);
   printf("2");
   printtypeshort(type2);
   printf(")");
}

struct _addr *widen ( struct _addr *A, int type )
{
   struct _addr *ap;

   if (A -> type == type) return A;
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = TEMP;
   ++tmpno; (ap -> value).temp = tmpno;
   ap -> type = type;
   printf("\t["); printtypeshort(type);
   printf("]\tt%d = ", tmpno); typeconv(A->type,type); printarg(A); printf("\n");
   return ap;
}

struct _addr *operation ( struct _addr *AA, char op, struct _addr *BB )
{
   struct _addr *A, *B, *ap;
   int lca;

   A = ( (AA -> cat == MEMLOC) || (AA -> cat == MEMLOCTEMP) ) ? memload(AA) : AA;
   B = ( (BB -> cat == MEMLOC) || (BB -> cat == MEMLOCTEMP) ) ? memload(BB) : BB;
   if (A -> type >= 4) yyerror("invalid first operand in arithmetic operation");
   if (B -> type >= 4) yyerror("invalid second operand in arithmetic operation");
   if ((op == '%') && ((A -> type >= 2) || (B -> type >= 2)) )
      yyerror("% operator on non-integer arguments");
   lca = LCA(A -> type, B -> type);
   A = widen(A,lca);
   B = widen(B,lca);
   printf("\t["); printtypeshort(lca);
   ++tmpno; printf("]\tt%d = ", tmpno); printarg(A); printf(" %c ", op); printarg(B); printf("\n");
   ap = (struct _addr *)malloc(sizeof(struct _addr));
   ap -> cat = TEMP;
   ap -> type = lca;
   (ap -> value).temp = tmpno;
   return ap;
}

void memstore ( struct _addr *lval, struct _addr *rval )
{
   if (lval -> type >= 4) yyerror("invalid type of l-value");
   if (rval -> type >= 4) yyerror("invalid type of r-value");
   if (rval -> cat == MEMLOCTEMP) {
      ++tmpno;
      printf("\t["); printtypeshort(rval -> type);
      printf("]\tt%d = MEM(t%d,%d)\n", tmpno, (rval -> value).toffset, TT[rval->type].width);
      rval -> cat = TEMP;
      (rval -> value).temp = tmpno;
   }
   if (lval -> type != rval -> type) {
      ++tmpno;
      printf("\t["); printtypeshort(lval -> type);
      printf("]\tt%d = ", tmpno); typeconv(rval->type,lval->type); printarg(rval); printf("\n");
      rval -> cat = MEMLOCTEMP;
      rval -> type = lval -> type;
      (rval -> value).toffset = tmpno;
   }
   if (lval -> cat == MEMLOCTEMP) {
      printf("\t["); printtypeshort(lval->type);
      printf("]\tMEM(t%d,%d) = ", (lval -> value).toffset, TT[lval -> type].width);
   }
   else if (lval -> cat == MEMLOC) {
      printf("\t["); printtypeshort(lval->type);
      printf("]\tMEM(%d,%d) = ", (lval -> value).offset, TT[lval -> type].width);
   } else yyerror("Bad l-value");
   if (rval -> cat == ICONST) printf("%d\n", (rval -> value).iconst);
   else if (rval -> cat == FCONST) printf("%.16lf\n", (rval -> value).fconst);
   else if (rval -> cat == TEMP) printf("t%d\n", (rval -> value).temp);
   else if (rval -> cat == MEMLOCTEMP) printf("t%d\n", (rval -> value).toffset);
   else printf("unknown\n");
   printf("\n");
}
