%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "defs.h"

typedef struct typeentry {
   int cat;            /* Basic types like CHR, UINT, FLT, DBL or
                          Pointer type PTR or
                          Array type ARR or
                          structure type STRUCT */
   int dim;            /* Relevant only for ARR type */
   int ref;            /* ARR: Element type (index in tyoe table)
                          PTR: Element type pointed to (index in type table)
                          STRUCT: Index of the symbol table for the structure */
   int width;          /* sizeof field */
   char *name;         /* Relevant only for STRUCT */
} ttentry;

ttentry TT[4096];      /* At most 4096 types (including the basic types) can be supported */
int nttentry;          /* Number of types defined so far */

typedef struct nameentry {
   char *name;         /* Name of the variable */
   int typeref;        /* Index in the type table */
   int offset;         /* Memory offset in the data segment or structure */
} stentry;

stentry ST[64][1024];  /* ST[0] is the main symbol table. ST[1], ST[2], ... are for structures */
int nstentry[64];      /* Count of variables in each symbol table */
int nst = 1;           /* Number of symbol tables in use */
int width[64];         /* Width of all variables in the symbol table */
int stref[64];         /* Reference in the type table */

struct _attr {
   int typeref;
   int stno;
};
%}

%union {
   int lexval;
   char *lexstr;
   struct _attr *attr;
   int stno;
   int typeref;
}

%token VOID UCHR CHR USRT SRT ULNG LNG UINT INT FLT DBL STRUCT ARR PTR
%token <lexval> NUM
%token <lexstr> ID

%type <stno> STN0 STN1 STN2 STN3
%type <typeref> BASIC
%type <attr> VT1 VT2 VT3 VT4 VT5 VT6 VT7 VT8 DIM

%%

PROG	: STN0 DECLIST
	;

DECLIST	: DECLIST STN1 DECL
	| STN2 DECL
	;

DECL	: BASIC VT1 LIST ';'
	| STRUCT ID '{' STN3 DECLIST WD '}' ';'
	| STRUCT ID '{' STN3 DECLIST WD '}' VT2 LIST ';'
	| STRUCT ID VT3 LIST ';'
	;

STN0	: { $$ = 0; } ;
STN1	: { $$ = $<stno>-1; } ;
STN2	: { $$ = $<stno>0; } ;
STN3	: { $$ = TT[addstructtype($<lexstr>-1)].ref; } ;

WD	: { TT[stref[$<stno>-1]].width = width[$<stno>-1]; } ;

BASIC	: VOID   { $$ = searchbasictype (VOID); }
	| UCHR   { $$ = searchbasictype (UCHR); }
	| CHR    { $$ = searchbasictype (CHR); }
	| USRT   { $$ = searchbasictype (USRT); }
	| SRT    { $$ = searchbasictype (SRT); }
	| ULNG   { $$ = searchbasictype (ULNG); }
	| LNG    { $$ = searchbasictype (LNG); }
	| UINT   { $$ = searchbasictype (UINT); }
	| INT    { $$ = searchbasictype (INT); }
	| FLT    { $$ = searchbasictype (FLT); }
	| DBL    { $$ = searchbasictype (DBL); }
	;

LIST	: LIST ',' VT4 VAR
	| VT5 VAR
	;

VAR	: '*' VT6 VAR
	| ID VT7 DIM   { STadd($1,$3->typeref,$3->stno); }
	;

DIM	: '[' NUM ']' VT8 DIM { $$ = genattr(addarraytype($2,$5->typeref),$5->stno); }
	| { $$ = $<attr>0; }
	;

VT1	: { $$ = genattr($<typeref>0, $<stno>-1); } ;
VT2	: { $$ = genattr(stref[$<stno>-3], $<stno>-7); } ;
VT3	: { $$ = genattr(searchstructtype($<lexstr>0), $<stno>-2); } ;
VT4	: { $$ = $<attr>-2; } ;
VT5	: { $$ = $<attr>0; } ;
VT6	: { $$ = genattr(addpointertype(($<attr>-1) -> typeref), ($<attr>-1) -> stno); } ;
VT7	: { $$ = $<attr>-1; } ;
VT8	: { $$ = $<attr>-3; } ;

%%

void yyerror (char *msg ) {
   fprintf(stderr, "*** %s\n", msg);
   exit(1);
}

int main ( int argc, char *argv[] )
{
   if (argc > 1) yyin = (FILE *)fopen(argv[1],"r");
   addbasictypes();
   yyparse();
   fclose(yyin);
   printalltypes();
   printallsymbols();
   exit(0);
}

void addbasictypes ( )
{
   TT[0].cat  = VOID; TT[0].width  = 0;
   TT[1].cat  = UCHR; TT[1].width  = sizeof(unsigned char);
   TT[2].cat  = CHR;  TT[2].width  = sizeof(char);
   TT[3].cat  = USRT; TT[3].width  = sizeof(unsigned short);
   TT[4].cat  = SRT;  TT[4].width  = sizeof(short);
   TT[5].cat  = ULNG; TT[5].width  = sizeof(unsigned long);
   TT[6].cat  = LNG;  TT[6].width  = sizeof(long);
   TT[7].cat  = UINT; TT[7].width  = sizeof(unsigned int);
   TT[8].cat  = INT;  TT[8].width  = sizeof(int);
   TT[9].cat  = FLT;  TT[9].width  = sizeof(float);
   TT[10].cat = DBL;  TT[10].width = sizeof(double);
   nttentry = 11;
}

int searchbasictype ( int type )
{
   switch (type) {
      case VOID : return 0;
      case UCHR : return 1;
      case CHR  : return 2;
      case USRT : return 3;
      case SRT  : return 4;
      case ULNG : return 5;
      case LNG  : return 6;
      case UINT : return 7;
      case INT  : return 8;
      case FLT  : return 9;
      case DBL  : return 10;
      default:
         fprintf(stderr, "*** Unrecognized basic type: %d\n", type);
         return -1;
   }
}

int addarraytype ( int dim, int ttref )
{
   int i;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == ARR) && (TT[i].ref == ttref) && (TT[i].dim == dim) ) return i;
   }
   i = nttentry;
   ++nttentry;
   TT[i].cat = ARR;
   TT[i].dim = dim;
   TT[i].ref = ttref;
   TT[i].width = dim * TT[ttref].width;
   return i;
}

int addpointertype ( int ttref )
{
   int i, w;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == PTR) && (TT[i].ref == ttref) ) return i;
   }
   i = nttentry;
   ++nttentry;
   TT[i].cat = PTR;
   TT[i].ref = ttref;
   TT[i].width = sizeof(void *);
   return i;
}

int searchstructtype ( char *name )
{
   int i;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == STRUCT) && (!strcmp(TT[i].name,name)) ) return i;
   }
   yyerror("Undefined structure name");
}

int addstructtype ( char *name )
{
   int i, w;

   for (i=0; i<nttentry; ++i) {
      if ( (TT[i].cat == STRUCT) && (!strcmp(TT[i].name,name)) ) {
         yyerror("Structure name already exists");
      }
   }
   i = nttentry;
   ++nttentry;
   TT[i].cat = STRUCT;
   TT[i].name = strdup(name);
   TT[i].ref = nst;
   stref[nst] = i;
   width[nst] = 0;
   ++nst;
   return i;
}

struct _attr *genattr ( int typeref, int stno )
{
   struct _attr *p;

   p = (struct _attr *)malloc(sizeof(struct _attr));
   p -> typeref = typeref;
   p -> stno = stno;
   return p;
}

void STadd ( char *name, int typeref, int t )
{
   int i, w;

   for (i=0; i<nstentry[t]; ++i) {
      if (!strcmp(ST[t][i].name,name)) yyerror("Duplicate variable name");
   }
   i = nstentry[t];
   ++(nstentry[t]);
   ST[t][i].name = strdup(name);
   ST[t][i].typeref = typeref;
   ST[t][i].offset = width[t];
   width[t] += TT[typeref].width;
   width[t] = ((width[t] + 3) / 4) * 4;
}

void printtype ( int i )
{
   if (TT[i].cat == ARR) {
      printf("array(%d,", TT[i].dim); printtype(TT[i].ref); printf(")");
   } else if (TT[i].cat == PTR) {
      printf("pointer("); printtype(TT[i].ref); printf(")");
   } else if (TT[i].cat  == STRUCT) {
      printf("struct %s with symbol table %d", TT[i].name, TT[i].ref);
   } else {
      printbasicname(TT[i].cat);
   }
}

void printalltypes ( )
{
   int i;

   printf("+++ All declarations read\n");
   printf("\n+++ %d types\n", nttentry);
   for (i=0; i<nttentry; ++i) {
      printf("    Type %3d: %8d    ", i, TT[i].width);
      printtype(i);
      printf("\n");
   }
}

void printbasicname ( int i )
{
   switch (i) {
      case VOID:  printf("void"); break;
      case UCHR:  printf("unsigned char"); break;
      case CHR :  printf("char"); break;
      case USRT:  printf("unsigned short"); break;
      case SRT :  printf("short"); break;
      case ULNG:  printf("unsigned long"); break;
      case LNG :  printf("long"); break;
      case UINT:  printf("unsigned int"); break;
      case INT :  printf("int"); break;
      case FLT :  printf("float"); break;
      case DBL :  printf("double"); break;
      default  : printf("unknown"); break;
   }
}

void printallsymbols ( )
{
   int i;

   for (i=0; i<nst; ++i) {
      printf("\n+++ Symbol table %d ", i);
      if (i) printf("[struct %s]\n", TT[stref[i]].name);
      else printf("[main]\n");
      printsymbols(i);
   }
}

void printsymbols ( int t )
{
   int i;

   for (i=0; i<nstentry[t]; ++i) {
      printf("    %-12s%10d - %-10d  type = %4d = ", ST[t][i].name, ST[t][i].offset,
             ST[t][i].offset + TT[ST[t][i].typeref].width - 1,
             ST[t][i].typeref);
      printtype(ST[t][i].typeref);
      printf("\n");
   }
   printf("    Total width = %d\n", width[t]);
}
